/*
 * PES file parsing.
 *
 * All format credit goes to Robert Heel and 
 *
 *	https://bugs.launchpad.net/inkscape/+bug/247463
 *
 * which has a php script to do so. He in turn seems to have
 * gotten it from NJ Crawford's C# PES viewer. I just turned
 * it into C.
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include "pes.h"


// These values are designed to be canonical in that they are the actual 
// values expected by the PES format. There are several assumptions in this though. 
// I have taken Linus's rgb values and matched them as close as I can to the Brother Embroidery trhread rgb values. 
// This gives good results for most, the PES pallete is very slightly smaller than the thread pallete, but 
// most of the given rgb values map to a thread with a low error vector, giving good confidence that the values are correct. 
// Most thread are picked exactly once. 
// 3 threads are picked twice, the last 3 in the list that have suspicious round number rgb values. 
// The names used are taken from the colors used, the aim is that the number and name should be what is displayed by
// a brother embroidery machine when sewing. I had a design that used both red and vermillion, well how should I know 
// which to use? But having the color number available and making my machine or this code tell me well at 
// least I could reproduce the original. Color are not always intuative, so beware. Like Dark grey really is pretty much 
// a black, and Cream Brown (010) looks like a very unsaturated shade of yellow or green to me. Thats why we need rgb values too. 
static struct color color_def[256] = {
	{ NULL, 0, 0, 0 },
	{ "Prussian Blue Brother, 007 (17,31,119) #5.8",		 14,  31, 124 },
	{ "Blue Brother, 405 (10,83,168) #5.4",		 10,  85, 163 },
	{ "Teal Green Brother, 534 (0,138,116) #48.2",		 48, 135, 119 },
	{ "Corn Flower Brother, 070 (80,104,183) #9.9",		 75, 107, 175 },
	{ "Red Brother, 800 (233,26,27) #6.4",		237,  23,  31 },
	{ "Reddish Brown Brother, 337 (204,93,3) #5.9",		209,  92,   0 },
	{ "Magenta Brother, 620 (143,56,144) #7.5",		145,  54, 151 },
	{ "Light Lilac Brother, 810 (228,154,204) #1.0",		228, 154, 203 },
	{ "Lilac Brother, 612 (146,94,175) #3.3",		145,  95, 172 },
	{ "Mint Green Brother, 502 (161,214,125) #4.0",		157, 214, 125 },
	{ "Deep Gold Brother, 214 (230,174,0) #5.4",		232, 169,   0 },
	{ "Orange Brother, 208 (252,187,52) #2.4",		254, 186,  53 },
	{ "Yellow Brother, 205 (255,255,2) #2.0",		255, 255,   0 },
	{ "Lime Green Brother, 513 (112,187,41) #10.0",		112, 188,  31 },
	{ "Brass Brother, 328 (192,149,9) #9.1",		192, 148,   0 },
	{ "Silver Brother, 005 (169,168,167) #1.4",		168, 168, 168 },
	{ "Russet Brown Brother, 330 (125,110,10) #10.2",		123, 111,   0 },
	{ "Cream Brown Brother, 010 (255,253,178) #2.2",		255, 255, 179 },
	{ "Pewter Brother, 704 (81,84,86) #2.2",		 79,  85,  86 },
	{ "Black Brother, 900 (0,0,0)",		  0,   0,   0 },
	{ "Ultramarine Brother, 406 (10,62,145) #1.4",		 11,  61, 145 },
	{ "Royal Purple Brother, 869 (115,5,121) #6.4",		119,   1, 118 },
	{ "Dark Gray Brother, 707 (42,49,51) #1.0",		 41,  49,  51 },
	{ "Dark Brown Brother, 058 (43,19,0) #1.4",		 42,  19,   1 },
	{ "Deep Rose Brother, 086 (240,77,141) #7.3",		246,  74, 138 },
	{ "Light Brown Brother, 323 (180,118,38) #2.8",		178, 118,  36 },
	{ "Salmon Pink Brother, 079 (254,184,201) #6.2",		252, 187, 196 },
	{ "Vermilion Brother, 030 (255,56,11) #4.2",		254,  55,  15 },
	{ "White Brother, 100 (240, 240, 240)", 240,240,240 },
	{ "Violet Brother, 613 (107,28,136) #2.2",		106,  28, 138 },
	{ "Seacrest Brother, 542 (172,219,192) #6.0",		168, 221, 196 },
	{ "Sky Blue Brother, 019 (33,135,190) #5.8",		 37, 132, 187 },
	{ "Pumpkin Brother, 126 (253,178,68) #1.7",		254, 179,  67 },
	{ "Cream Yellow Brother, 812 (255,238,145) #4.5",		255, 240, 141 },
	{ "Khaki Brother, 348 (212,165,94) #4.6",		208, 166,  96 },
	{ "Clay Brown Brother, 339 (217,84,0) #8.0",		209,  84,   0 },
	{ "Leaf Green Brother, 509 (105,185,69) #5.1",		102, 186,  73 },
	{ "Peacock Blue Brother, 415 (18,73,74) #4.2",		 19,  74,  70 },
	{ "Gray Brother, 817 (137,136,135) #2.2",		135, 135, 135 },
	{ "Warm Gray Brother, 399 (216,204,201) #3.6",		216, 202, 198 },
	{ "Dark Olive Brother, 517 (67,82,10) #5.0",		 67,  86,   7 },
	{ "Linen Brother, 307 (255,227,196) #1.4",		254, 227, 197 },
	{ "Pink Brother, 085 (249,148,185) #3.2",		249, 147, 188 },
	{ "Deep Green Brother, 808 (4,55,39) #6.5",		  0,  56,  34 },
	{ "Lavender Brother, 804 (182,174,216) #5.7",		178, 175, 212 },
	{ "Wistaria Violet Brother, 607 (104,104,181) #5.4",		104, 106, 176 },
	{ "Beige Brother, 843 (239,227,188) #3.0",		239, 227, 185 },
	{ "Carmine Brother, 807 (250,55,96) #6.8",		247,  56, 102 },
	{ "Amber Red Brother, 333 (180,78,101) #2.4",		181,  76, 100 },
	{ "Olive Green Brother, 519 (19,43,24) #2.0",		 19,  43,  26 },
	{ "Dark Fuchsia Brother, 107 (199,3,86) #2.2",		199,   1,  85 },
	{ "Tangerine Brother, 209 (251,157,50) #3.2",		254, 158,  50 },
	{ "Light Blue Brother, 017 (174,219,229) #9.0",		168, 222, 235 },
	{ "Emerald Green Brother, 507 (5,103,63) #37.3",		  0, 103,  26 },
	{ "Purple Brother, 614 (79,41,141) #3.2",		 78,  41, 144 },
	{ "Moss Green Brother, 515 (48,125,38) #6.2",		 47, 126,  32 },
	{ "Flesh Pink Brother, 124 (250,218,222) #3.2",		253, 217, 222 },
	{ "Harvest Gold Brother, 206 (253,219,17) #2.8",		255, 217,  17 },
	{ "Electric Blue Brother, 420 (12,91,165) #3.2",		  9,  91, 166 },
	{ "Lemon Yellow Brother, 202 (237,250,115) #4.4",		240, 249, 112 },
	{ "Fresh Green Brother, 027 (227,242,96) #5.1",		227, 243,  91 },
// The 3 below values are all duplicates and values seem suspect (round numbers). Maybe they are just for drawing purposes. 
	{ "Pumpkin Brother, 126 (253,178,68) #38.9",		255, 200, 100 }, 
    // to my eyes 206 would be better match, but the rgb values give a medium yellow. Nothing orange bout it.  

	{ "Cream Yellow Brother, 812 (255,238,145) #38.3",		255, 200, 150 }, // v pale orange 
	{ "Salmon Pink Brother, 079 (254,184,201) #16.1",		255, 200, 200 }, // v pale pink 
};

static struct color *my_colors[256];

#define CHUNKSIZE (8192)

int read_file(int fd, struct region *region)
{
	int len = 0, done = 0;
	char *buf = NULL;

	for (;;) {
		int space = len - done, ret;
		if (!space) {
			space = CHUNKSIZE;
			len += space;
			buf = realloc(buf, len);
		}
		ret = read(fd, buf + done, space);
		if (ret > 0) {
			done += ret;
			continue;
		}
		if (!ret)
			break;
		if (errno == EINTR || errno == EAGAIN)
			continue;
		free(buf);
		return -1;
	}

	/* "len+8" guarantees that there is some slop at the end */
	region->ptr = realloc(buf, len+8);
	region->size = len;
	return 0;
}

int read_path(const char *path, struct region *region)
{
	if (path) {
		int fd = open(path, O_RDONLY);
		if (fd > 0) {
			int ret = read_file(fd, region);
			int saved_errno = errno;
			close(fd);
			errno = saved_errno;
			return ret;
		}
		return fd;
	}
	return read_file(0, region);
}

#define get_u8(buf, offset) (*(unsigned char *)((offset)+(const char *)(buf)))
#define get_le32(buf, offset) (*(unsigned int *)((offset)+(const char *)(buf)))

static int parse_pes_colors(struct region *region, unsigned int pec)
{
	const void *buf = region->ptr;
	int nr_colors = get_u8(buf, pec+48) + 1;
	int i;

	for (i = 0; i < nr_colors; i++) {
		struct color *color;
		color = color_def + get_u8(buf, pec+49+i);
		my_colors[i] = color;
	}
	return 0;
}

static struct pes_block *new_block(struct pes *pes)
{
	struct pes_block *block = calloc(1, sizeof(*block));

	if (block) {
		struct pes_block **pp = pes->last ? &pes->last->next : &pes->blocks;
		*pp = block;
		pes->last = block;
		block->color = my_colors[pes->nr_colors++];
	}
	return block;
}

static int add_stitch(struct pes *pes, int x, int y, int jumpstitch)
{
	struct pes_block *block = pes->last;
	struct stitch *stitch = block->stitch;
	int nr_stitches = block->nr_stitches;

	if (x < pes->min_x)
		pes->min_x = x;
	if (x > pes->max_x)
		pes->max_x = x;
	if (y < pes->min_y)
		pes->min_y = y;
	if (y > pes->max_y)
		pes->max_y = y;

	if (block->max_stitches == nr_stitches) {
		int new_stitches = (nr_stitches * 3) / 2 + 64;
		int size = new_stitches*sizeof(struct stitch);
		stitch = realloc(stitch, size);
		if (!stitch)
			return -1;
		block->max_stitches = new_stitches;
		block->stitch = stitch;
	}
	stitch[nr_stitches].x = x;
	stitch[nr_stitches].y = y;
	stitch[nr_stitches].jumpstitch = jumpstitch;
	block->nr_stitches = nr_stitches+1;
	return 0;
}

static int parse_pes_stitches(struct region *region, unsigned int pec, struct pes *pes)
{
	int oldx, oldy;
	const unsigned char *buf = region->ptr, *p, *end;
	struct pes_block *block;

	p = buf + pec + 532;
	end = buf + region->size;

	oldx = oldy = 0;

	block = new_block(pes);

	while (p < end) {
		int val1 = p[0], val2 = p[1], jumpstitch = 0;
		p += 2;
		if (val1 == 255 && !val2)
			return 0;
		if (val1 == 254 && val2 == 176) {
			if (block->nr_stitches) {
				block = new_block(pes);
				if (!block)
					return -1;
			}
			p++;	/* Skip byte */
			continue;
		}

		/* High bit set means 12-bit offset, otherwise 7-bit signed delta */
		if (val1 & 0x80) {
			val1 = ((val1 & 15) << 8) + val2;
			/* Signed 12-bit arithmetic */
			if (val1 & 2048)
				val1 -= 4096;
			val2 = *p++;
			jumpstitch = 1;
		} else {
			if (val1 & 64)
				val1 -= 128;
		}

		if (val2 & 0x80) {
			val2 = ((val2 & 15) << 8) + *p++;
			/* Signed 12-bit arithmetic */
			if (val2 & 2048)
				val2 -= 4096;
			jumpstitch = 1;
		} else {
			if (val2 & 64)
				val2 -= 128;
		}

		val1 += oldx;
		val2 += oldy;

		oldx = val1;
		oldy = val2;

		if (add_stitch(pes, val1, val2, jumpstitch))
			return -1;
	}
	return 0;
}

int parse_pes(struct region *region, struct pes *pes)
{
	const void *buf = region->ptr;
	unsigned int size = region->size;
	unsigned int pec;

	if (size < 48)
		return -1;
	if (memcmp(buf, "#PES", 4))
		return -1;
	pec = get_le32(buf, 8);
	if (pec > region->size)
		return -1;
	if (pec + 532 >= size)
		return -1;
	if (parse_pes_colors(region, pec) < 0)
		return -1;
	return parse_pes_stitches(region, pec, pes);
}
