/* JEFScan.c  JEF file scanner */
/* 2009-09-28 Robert Forsyth
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <stdarg.h>
#include <cairo/cairo.h>

#include "JEF.h"

int jef2png(const char *input,const char *output,int outputsize,double density); 
void color3(char *s,double *r,double *g,double *b); 

static void report(const char *fmt, va_list params)
{
	vfprintf(stderr, fmt, params);
}

static void die(const char *fmt, ...)
{
	va_list params;

	va_start(params, fmt);
	report(fmt, params);
	va_end(params);
	exit(1);
}

double bg=-1; // transparent; 
int
main( int argc, char *argv[])
{
	int i; 
	double density = 1.0;
    int outputsize = 128;

	const char *output = NULL;
	const char *input = NULL;

    if (argc < 3)
       die("usage: %s [-s n] [-d n] file.jef file.png \n", argv[0]);
	for (i = 1; i < argc; i++) 
     {
		const char *arg = argv[i];
		if (*arg == '-') {
			switch (arg[1]) {
			case 's':
				outputsize = atoi(argv[i+1]);
				i++;
				continue;
			case 'd':
				density = atof(argv[i+1]);
				i++;
				continue;
            case 'w':  // white background 
                bg=1.0; // well nearly white
                continue; 
			}
			die("Unknown argument '%s'\n", arg);
		}
		if (!input) {
			input = arg;
			continue;
        } 
		if (!output) {
			output = arg;
			continue;
        } 
    } 
	if (!output)
		die("Need a png output file name\n");
	if (!input)
		die("Need a jef input file name\n");

    // printinfo(input); 
    int r=jef2png(input,output,outputsize,density);  
    if (r==0) exit(1); 

  return EXIT_SUCCESS;
}

int jef2png(const char *input,const char *output,int outputsize,double density)
{
	cairo_surface_t *surface;
	cairo_t *cr;
	int reachedEnd = 0;

    JEF_t *aJEF = JEFNewPath( JEFAlloc(), input);
    
    if (aJEF)
    {
      long i;
      long x = 0, y = 0;
      long lft = 0, rgt = 0, top = 0, btm = 0;
      //double threadLen = 0.0;
      // const JEFHoop_t *aHoop = NULL;
      //aHoop = JEFHoop( aJEF->hoopCode);
      //if (aJEF->JEFThreadColours && aJEF->JEFThreadTypes)
      //for (i=0; i<aJEF->threadCount; ++i)
 	  //{
	    //const JEFColour_t *aJEFColour = JEFColour(aJEF->JEFThreadColours[i]);
	    // printf( "Thread %ld: 0x%02lx %ld %s %s %s\n", (long)i + 1, (long)aJEF->JEFThreadColours[i], (long)aJEF->JEFThreadTypes[i], aJEFColour->ColourNumber, aJEFColour->RGB, aJEFColour->ColourName);
	  //}
      if (aJEF->JEFThreadColours && aJEF->JEFThreadTypes && aJEF->JEFStitches)
      {
	    int moveWithoutStitch = 0;
	    long threadIndex = 0;
	    //printf( "Load Thread %ld: %ld %ld\n", (long)threadIndex + 1, (long)aJEF->JEFThreadColours[threadIndex], (long)aJEF->JEFThreadTypes[threadIndex]);
	    for (i=0; i<aJEF->stitchCount; ++i)
	    {
	      JEFStitch_t s = aJEF->JEFStitches[i];
	      if (s.dx == -128)
	      {
	        switch (s.dy)
	        {
	          case 1: /* change thread */
		        ++threadIndex;
		        if (threadIndex >= aJEF->threadCount)
		           threadIndex = 0;
		        //threadLen = 0.0;
		      break;
	          case 2: /* move */
		        moveWithoutStitch = 1;
		      break;
	          case 16: /* end */
		        reachedEnd = 1;
		      break;
	        }
	      }
	      else
	      {
	        x += s.dx;
	        y += s.dy;
	        if (x < lft) lft = x;
	        if (x > rgt) rgt = x;
	        if (y < btm) btm = y;
	        if (y > top) top = y;
	        moveWithoutStitch = 0;
	      }
	    }

        double r,g,b; 
        double scale=0.0; 
        int outw,outh;
        int width, height; 
        width=rgt-lft;
        height=top-btm;
        x=0.0;y=0.0;threadIndex=0;
    
 	    int maxd = width > height ? width : height;
	    scale = (double) outputsize / maxd;
	    outw = width * scale;
	    outh = height * scale;
	    surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, outw, outh);
	    cr = cairo_create (surface);
        if (bg>=0)
        {
          cairo_rectangle(cr, 0.0, 0.0, outw,outh);
	      cairo_set_source_rgb(cr, bg,bg,bg);
          cairo_fill(cr); 
        } 
        {
	      const JEFColour_t *aJEFColour = JEFColour(aJEF->JEFThreadColours[threadIndex]);
          color3(aJEFColour->RGB,&r,&g,&b); 
	      cairo_set_source_rgb(cr, r,g,b);
          cairo_set_line_width(cr, scale * density);
        }
        reachedEnd=0;

	    for (i=0; i<aJEF->stitchCount; ++i)
	    {
	      JEFStitch_t s = aJEF->JEFStitches[i];
	      if (s.dx == -128)
	      {
	        switch (s.dy)
	        {
	          case 1: /* change thread */
		        ++threadIndex;
		        if (threadIndex >= aJEF->threadCount)
		           threadIndex = 0;
		        // printf( "Load Thread %ld: %ld %ld\n", (long)threadIndex + 1, (long)aJEF->JEFThreadColours[threadIndex], (long)aJEF->JEFThreadTypes[threadIndex]);
		        //threadLen = 0.0;
	            const JEFColour_t *aJEFColour = JEFColour(aJEF->JEFThreadColours[threadIndex]);
                color3(aJEFColour->RGB,&r,&g,&b); 
		cairo_stroke(cr);
	            cairo_set_source_rgb(cr, r,g,b);
		      break;
	          case 2: /* move */
	//	cairo_stroke(cr);
		        moveWithoutStitch = 1;
		      break;
	          case 16: /* end */
		        reachedEnd = 1;
		//cairo_set_line_width(cr, scale * density);
		//cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);
		//cairo_set_line_join(cr, CAIRO_LINE_JOIN_ROUND);
		cairo_stroke(cr);
cairo_surface_write_to_png(surface, output);
		      break;
	        }
	      }
	      else
	      {
	        x += s.dx;
	        y += s.dy;
	        //if (x < lft) lft = x;
	        //if (x > rgt) rgt = x;
	        //if (y < btm) btm = y;
	        //if (y > top) top = y;
	        if (!moveWithoutStitch)
	        {
	          //modLen = sqrt( s.dx * s.dx + s.dy * s.dy);
	          //threadLen += modLen;
			  cairo_line_to(cr, (x-lft)*scale,(top-y)*scale);
	        }
            else
            {
		        cairo_move_to(cr, (x-lft)*scale,(top-y)*scale);
            }
	        moveWithoutStitch = 0;
	      }
	    }
	    // printf( "Reached end: %s\n", (char*) (reachedEnd ? "Yes" : "No"));
       }
    }
    aJEF = JEFFree( aJEF);
    return reachedEnd; 
}
void color3(char *s,double *r,double *g,double *b)
{
  // printf("%s ",s); 
  *s=='#' && s++; // skip the #
  int ir,ig,ib; 
  sscanf(s,"%02X%02X%02X",&ir,&ig,&ib); 
  // apply any contrast factors. w, b. 
  // a w factor of less than 1.0 does not affect blacks, but reduces full white luminosity multiplicatly. 
  // eg w=0.9 means whites become 90% white. 
  // b factor does the same with blacks, does not affect full whites, but lifts blacks proportionally. 
  // can use both 
  // double, b, w; 
  const double wf=0.9; // this means designs with white can always be displayed on a true white background. 
  const double bf=1.0; // and if you need to display on a full black background, set this down from 1.0 

  *r=wf*ir/255.0; 
  *g=wf*ig/255.0; 
  *b=wf*ib/255.0; 
  *r=(1-(1-*r)*bf); 
  *g=(1-(1-*g)*bf); 
  *b=(1-(1-*b)*bf); 
  // printf(" (%f,%f,%f)\n",*r,*b,*g); 
} 
  
int printinfo(char *input)
{
    JEF_t *aJEF = JEFNewPath( JEFAlloc(), input);
    
    printf( "JEF file %s\n", input);
    if (aJEF)
    {
      long i;
      long x = 0, y = 0;
      long lft = 0, rgt = 0, top = 0, btm = 0;
      double threadLen = 0.0;
      const JEFHoop_t *aHoop = NULL;
      printf( "Header Length: %ld\n", (long) aJEF->stitchesSeek);
      printf( "Flags: %04lx\n", (long) aJEF->flags);
      printf( "Date/time: %s\n", (char*) aJEF->datetime);
      printf( "Thread count: %ld\n", (long) aJEF->threadCount);
      printf( "Stitch count: %ld\n", (long) aJEF->stitchCount);
      aHoop = JEFHoop( aJEF->hoopCode);
      printf( "Hoop: %ld", (long) aJEF->hoopCode);
      if (aHoop)
	      printf( " %s %ld %ld", aHoop->hoopType, aHoop->hoopWidth, aHoop->hoopHeight);
      printf( "\n");
      printf( "Rectangle 1: %ld %ld %ld %ld\n", (long) aJEF->left1, (long) aJEF->top1, (long) aJEF->right1, (long) aJEF->bottom1);
      printf( "Rectangle 2: %ld %ld %ld %ld\n", (long) aJEF->left2, (long) aJEF->top2, (long) aJEF->right2, (long) aJEF->bottom2);
      printf( "Rectangle 3: %ld %ld %ld %ld\n", (long) aJEF->left3, (long)      aJEF->top3, (long) aJEF->right3, (long) aJEF->bottom3);
      printf( "Rectangle 4: %ld %ld %ld %ld\n", (long) aJEF->left4, (long) aJEF->top4, (long) aJEF->right4, (long) aJEF->bottom4);
      printf( "Rectangle 5: %ld %ld %ld %ld\n", (long) aJEF->left5, (long) aJEF->top5, (long) aJEF->right5, (long) aJEF->bottom5);
      if (aJEF->JEFThreadColours && aJEF->JEFThreadTypes)
	for (i=0; i<aJEF->threadCount; ++i)
	{
	  const JEFColour_t *aJEFColour = JEFColour(aJEF->JEFThreadColours[i]);
	  printf( "Thread %ld: 0x%02lx %ld %s %s %s\n", (long)i + 1, (long)aJEF->JEFThreadColours[i], (long)aJEF->JEFThreadTypes[i], aJEFColour->ColourNumber, aJEFColour->RGB, aJEFColour->ColourName);
	}
      if (aJEF->JEFThreadColours && aJEF->JEFThreadTypes && aJEF->JEFStitches)
      {
	int moveWithoutStitch = 0;
	int reachedEnd = 0;
	long threadIndex = 0;
	printf( "Load Thread %ld: %ld %ld\n", (long)threadIndex + 1, (long)aJEF->JEFThreadColours[threadIndex], (long)aJEF->JEFThreadTypes[threadIndex]);
	for (i=0; i<aJEF->stitchCount; ++i)
	{
	  JEFStitch_t s = aJEF->JEFStitches[i];
	  if (s.dx == -128)
	  {
	    switch (s.dy)
	    {
	      case 1: /* change thread */
		printf( "Thread Used: %.1f\n", threadLen);
		++threadIndex;
		if (threadIndex >= aJEF->threadCount)
		  threadIndex = 0;
		printf( "Load Thread %ld: %ld %ld\n", (long)threadIndex + 1, (long)aJEF->JEFThreadColours[threadIndex], (long)aJEF->JEFThreadTypes[threadIndex]);
		threadLen = 0.0;
		break;
	      case 2: /* move */
		moveWithoutStitch = 1;
		break;
	      case 16: /* end */
		reachedEnd = 1;
		printf( "Thread Used: %.1f\n", threadLen);
		break;
	    }
	  }
	  else
	  {
	    double modLen;
	    x += s.dx;
	    y += s.dy;
	    if (x < lft) lft = x;
	    if (x > rgt) rgt = x;
	    if (y < btm) btm = y;
	    if (y > top) top = y;
	    if (!moveWithoutStitch)
	    {
	      modLen = sqrt( s.dx * s.dx + s.dy * s.dy);
	      threadLen += modLen;
	    }
	    moveWithoutStitch = 0;
	  }
	}
	printf( "Reached end: %s\n", (char*) (reachedEnd ? "Yes" : "No"));
      }
      printf( "Stitch extent: %ld %ld %ld %ld\n", (long) lft, (long)      top, (long) rgt, (long) btm);
      
    }
    
    aJEF = JEFFree( aJEF);
    return 0; 
}
