/* JEF.c - Janome Embroidery Format */
/* 2009-09-28 Robert Forsyth
 */

#include <stdio.h>
#include <fcntl.h> // for open
#include <unistd.h> // for close
#include "JEF.h"

#define DOBB(s) {fprintf( stderr, (s));}
#undef DOBB
#define DOBB(s) {;}

static JEFHoop_t JEFHoops[] =
{
//code type w h (0.1mm)
  {  0, "A", 1260, 1100},
  {  1, "C", 500, 500},
  {   2, "B", 1400, 2000} ,
  {  -1, "", 0, 0} 
};

const JEFHoop_t *JEFHoop(long aHoopCode)
{
  int i = 0;
  while (JEFHoops[i].JEFHoopCode != aHoopCode)
  {
    if (JEFHoops[i].JEFHoopCode == -1)
      break;
    ++i;
  }
  return JEFHoops + i;
}

JEF_t *JEFAlloc() /* return an empty JEF object or NULL */
{
  JEF_t *aJEF = malloc( sizeof (JEF_t));
  
  if (aJEF)
  {
    int i;
    
    /* just make it safe */
    aJEF->stitchesSeek = 0;
    aJEF->flags = 0;
    for (i = 0; i < (sizeof aJEF->datetime / sizeof (char)); ++i)
      aJEF->datetime[i] = 0;
    aJEF->threadCount = 0;
    aJEF->stitchCount = 0;
    aJEF->JEFThreadColours = NULL;
    aJEF->JEFThreadTypes = NULL;
    aJEF->JEFStitches = NULL;
  }
  return aJEF;
}

JEF_t *JEFFree( JEF_t *aJEF) /* return NULL after freeing JEF object */
{
  if (aJEF)
  {
    free( aJEF->JEFThreadColours); aJEF->JEFThreadColours = NULL;
    free( aJEF->JEFThreadTypes); aJEF->JEFThreadTypes = NULL;
    free( aJEF->JEFStitches); aJEF->JEFStitches = NULL;
  }
  free( aJEF); aJEF = NULL;
  return aJEF;
}

JEF_t *JEFNew( JEF_t *aJEF) /* initialise a JEF object - it is safe to do aJEF = JEFNew(JEFAlloc()) */
{
  int i;
  if (aJEF)
  {
    aJEF->stitchesSeek = 0;
    aJEF->flags = 0;
    for (i = 0; i < (sizeof aJEF->datetime / sizeof (char)); ++i)
      aJEF->datetime[i] = 0;
    aJEF->threadCount = 0;
    aJEF->stitchCount = 0;
    aJEF->hoopCode = 0;
    aJEF->left1 = aJEF->bottom1 = aJEF->right1 = aJEF->top1 = 0;
    aJEF->left2 = aJEF->bottom2 = aJEF->right2 = aJEF->top2 = 0;
    aJEF->left3 = aJEF->bottom3 = aJEF->right3 = aJEF->top3 = 0;
    aJEF->left4 = aJEF->bottom4 = aJEF->right4 = aJEF->top4 = 0;
    aJEF->left5 = aJEF->bottom5 = aJEF->right5 = aJEF->top5 = 0;
    aJEF->JEFThreadColours = NULL;
    aJEF->JEFThreadTypes = NULL;
    aJEF->JEFStitches = NULL;
  }
  return aJEF;
}

/* read four bytes as little-edian - return 0 on error */
static long readLong( int fd)
{
  long r = 0;
  unsigned char x[4];
  ssize_t br = read( fd, x, 4);
  if (br == 4)
  {
    r = (x[3] & 0x80) ? -1 : 0;
    r <<= 8; r |= x[3];
    r <<= 8; r |= x[2];
    r <<= 8; r |= x[1];
    r <<= 8; r |= x[0];
  }
  return r;
}

JEF_t *JEFNewFD( JEF_t *aJEF, int fd) /* initialise a JEF object from file descriptor */
{
  DOBB("JEFNewFD\n")
  if (aJEF)
  {
    long i;
    long p = 0;
    if (fd < 0) return JEFFree( aJEF);
    aJEF->stitchesSeek = readLong( fd);
    aJEF->flags = readLong( fd);
    if ((sizeof aJEF->datetime / sizeof (char)) != read( fd, aJEF->datetime, (sizeof aJEF->datetime / sizeof (char))))
      for (i = 0; i < (sizeof aJEF->datetime / sizeof (char)); ++i)
	aJEF->datetime[i] = 0;
    aJEF->datetime[15] = 0;
    aJEF->threadCount = readLong( fd);
    aJEF->stitchCount = readLong( fd);
    aJEF->hoopCode = readLong( fd);
    aJEF->left1 = -readLong( fd);
    aJEF->top1 = readLong( fd);
    aJEF->right1 = readLong( fd);
    aJEF->bottom1 = -readLong( fd);
    aJEF->left2 = -readLong( fd);
    aJEF->top2 = readLong( fd);
    aJEF->right2 = readLong( fd);
    aJEF->bottom2 = -readLong( fd);
    aJEF->left3 = -readLong( fd);
    aJEF->top3 = readLong( fd);
    aJEF->right3 = readLong( fd);
    aJEF->bottom3 = -readLong( fd);
    aJEF->left4 = -readLong( fd);
    aJEF->top4 = readLong( fd);
    aJEF->right4 = readLong( fd);
    aJEF->bottom4 = -readLong( fd);
    aJEF->left5 = -readLong( fd);
    aJEF->top5 = readLong( fd);
    aJEF->right5 = readLong( fd);
    aJEF->bottom5 = -readLong( fd);
    
    p = 0x74;
    /* this bit is slightly unsafe - we don't know if the counts are valid */
    /* stupidly I do not have anyway to detect early end-of-file or partial dribs and drabs reads */
    aJEF->JEFThreadColours = malloc( sizeof (JEFThreadColour_t) * aJEF->threadCount);
    for (i = 0; i < aJEF->threadCount; ++i)
    {
      JEFThreadColour_t x = readLong( fd);
      p += 4;
      if (aJEF->JEFThreadColours)
	aJEF->JEFThreadColours[i] = x;
    }
    aJEF->JEFThreadTypes = malloc( sizeof (JEFThreadType_t) * aJEF->threadCount);
    for (i = 0; i < aJEF->threadCount; ++i)
    {
      JEFThreadType_t x = readLong( fd);
      p += 4;
      if (aJEF->JEFThreadTypes)
	aJEF->JEFThreadTypes[i] = x;
    }
    while (p < aJEF->stitchesSeek)
    {
      fprintf(stderr, "File Position Error\n");
      char c;
      read( fd, &c, 1);
      ++p;
    }
    aJEF->JEFStitches = malloc( sizeof (JEFStitch_t) * aJEF->stitchCount);
    for (i = 0; i < aJEF->stitchCount; ++i)
    {
      // JEFStitch_t s;
      signed char c[2];
      if (2 != read( fd, c, 2))
	c[0] = c[1] = 0;
      p += 2;
      if (aJEF->JEFStitches)
      {
	aJEF->JEFStitches[i].dx = c[0];
	aJEF->JEFStitches[i].dy = c[1];
      }
    }
  }
  return aJEF;
}

JEF_t *JEFNewPath( JEF_t *aJEF, const char *path) /* initialise a JEF object from file */
{
  DOBB("JEFNewPath\n")
  int fd = open(path, O_RDONLY);

  aJEF = JEFNewFD( aJEF, fd);
  close(fd); fd = -1;
  return aJEF;
}

